/**
 * Background Service Worker
 * Handles badge updates and context menu for labeling articles
 */

const API_BASE = 'https://news-detox-extension-production.up.railway.app';
const EXTENSION_VERSION = '2.0.0';

// Set badge styling and create context menu on install
chrome.runtime.onInstalled.addListener(() => {
  chrome.action.setBadgeBackgroundColor({ color: '#c17f59' });

  // Create context menu for labeling articles
  chrome.contextMenus.create({
    id: 'label-block',
    title: '🌲 Označi kao propagandu',
    contexts: ['link', 'selection', 'page'],
    documentUrlPatterns: [
      '*://rs.n1info.com/*',
      '*://n1info.rs/*',
      '*://*.n1info.com/*'
    ]
  });
});

// Listen for messages from content scripts
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'updateBadge') {
    const count = request.count || 0;
    const text = count > 0 ? (count > 99 ? '99+' : count.toString()) : '';

    // Update badge for the specific tab
    if (sender.tab?.id) {
      chrome.action.setBadgeText({
        text: text,
        tabId: sender.tab.id
      });
    }

    sendResponse({ success: true });
  }

  if (request.action === 'submitLabel') {
    // Handle label submission from content script
    submitLabelToServer(request.data)
      .then(result => sendResponse(result))
      .catch(error => sendResponse({ success: false, error: error.message }));
    return true; // Keep message channel open for async response
  }

  return true;
});

// Handle context menu clicks
chrome.contextMenus.onClicked.addListener((info, tab) => {
  if (info.menuItemId === 'label-block' && tab?.id) {
    // Send message to content script to get article data
    chrome.tabs.sendMessage(tab.id, {
      action: 'getArticleForLabeling',
      linkUrl: info.linkUrl,
      selectionText: info.selectionText,
      pageUrl: info.pageUrl
    }, async (response) => {
      if (chrome.runtime.lastError) {
        console.log('[Šumski Filter] Content script not ready:', chrome.runtime.lastError.message);
        showNotification(tab.id, 'Greška: osvežite stranicu i pokušajte ponovo.');
        return;
      }

      if (response?.article) {
        const result = await submitLabelToServer(response.article);
        if (result.success) {
          showNotification(tab.id, '✓ Hvala! Članak je označen.');
        } else {
          showNotification(tab.id, result.message || 'Greška pri slanju.');
        }
      } else {
        showNotification(tab.id, 'Nije pronađen članak za označavanje.');
      }
    });
  }
});

// Submit label to server
async function submitLabelToServer(article) {
  try {
    const response = await fetch(`${API_BASE}/api/labels`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        title: article.title,
        description: article.description || null,
        url: article.url,
        labelType: 'block',
        extensionVersion: EXTENSION_VERSION
      })
    });

    const data = await response.json();

    if (response.ok) {
      return { success: true, message: data.message };
    } else {
      return { success: false, message: data.error || 'Server error' };
    }
  } catch (error) {
    console.error('[Šumski Filter] Label submission error:', error);
    return { success: false, message: 'Network error' };
  }
}

// Show notification in the page
function showNotification(tabId, message) {
  chrome.tabs.sendMessage(tabId, {
    action: 'showNotification',
    message: message
  });
}

// Clear badge when tab is updated/navigated
chrome.tabs.onUpdated.addListener((tabId, changeInfo) => {
  if (changeInfo.status === 'loading') {
    chrome.action.setBadgeText({ text: '', tabId: tabId });
  }
});
