/**
 * Bundled Content Script - No ES modules
 * All code in one file for Chrome extension compatibility
 */

// ========== CONFIGURATION ==========
const API_URL = 'https://news-detox-extension-production.up.railway.app';
const SEMANTIC_ENABLED = true;
const KEYWORD_BLOCK_THRESHOLD = 70;  // Keywords block at this score
const SEMANTIC_CHECK_THRESHOLD = 10; // Send to semantic if score is below keyword threshold but above this

// ========== NATURE IMAGES ==========
// Beautiful nature images from Serbia - Unsplash (free to use)
const NATURE_IMAGES = [
  "https://images.unsplash.com/photo-1735701392253-71e1c9b07d6d?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1735701392277-2693ff7b80a8?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1609858769940-bc9afdb4c4ae?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1698607265822-19dfbff919e3?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1709151783757-1c96b9687b57?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1746538175255-4b4b9f2ef89c?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1715476491784-e3b312fff5a0?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1750342625361-f2e88109c363?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1596525951480-505645fae86a?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1583141061799-af256b212f95?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1703356130913-874b82c2ada2?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1585337932029-d06cf6ffc75b?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1585338129664-c573a9554a61?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1585337931905-5289c506a080?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1708436229975-d5d4b6680ead?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1512904941170-5a699ee71295?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1742211155258-6b5ea7e437fe?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1742211132817-12ba314785b7?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1742209998726-5ed175d3ae02?auto=format&fit=crop&w=1200&q=80",
  "https://images.unsplash.com/photo-1742210198928-d2a8781d03b8?auto=format&fit=crop&w=1200&q=80"
];

// ========== SERBIAN WORD BOUNDARY HELPER ==========
// JavaScript \b doesn't work with Serbian characters (č, ć, ž, š, đ)
// Use (?<![a-zA-ZčćžšđČĆŽŠĐ]) for start and (?![a-zA-ZčćžšđČĆŽŠĐ]) for end
const SRB_START = '(?<![a-zA-ZčćžšđČĆŽŠĐ])'; // Negative lookbehind
const SRB_END = '(?![a-zA-ZčćžšđČĆŽŠĐ])';   // Negative lookahead

// Helper to create Serbian-aware word boundary regex
function srbWord(word) {
  return new RegExp(SRB_START + word + SRB_END, 'i');
}

// ========== KEYWORDS DATABASE ==========
// Comprehensive propaganda and political noise detection
const KEYWORDS_DB = {
  // Tier 1: Core regime figures - immediate high score
  tier1: [
    // President and closest associates
    { pattern: srbWord('vučić'), weight: 100, category: 'politician' },
    { pattern: /aleksand(?:a|)r\s+vučić/i, weight: 100, category: 'politician' },
    { pattern: /predsednik\s+(srbije|vučić|rekao|izjavio|najavio|poručio)/i, weight: 95, category: 'politician' },
    
    // Party
    { pattern: /\bsns\b/i, weight: 90, category: 'party' },
    { pattern: /srpska\s+napredna\s+stranka/i, weight: 100, category: 'party' },
    { pattern: srbWord('naprednjaci?'), weight: 85, category: 'party' },
    
    // Key government figures
    { pattern: srbWord('brnabić'), weight: 85, category: 'politician' },
    { pattern: /ana\s+brnabić/i, weight: 90, category: 'politician' },
    { pattern: srbWord('vučević'), weight: 85, category: 'politician' },
    { pattern: /miloš\s+vučević/i, weight: 90, category: 'politician' },
    { pattern: srbWord('dačić'), weight: 80, category: 'politician' },
    { pattern: /ivica\s+dačić/i, weight: 85, category: 'politician' }
  ],

  // Tier 2: Extended regime network
  tier2: [
    // Ministers and officials
    { pattern: /\bmali\b/i, weight: 55, category: 'politician', context: ['ministar', 'finansij', 'siniša', 'vlad'] },
    { pattern: /siniša\s+mali/i, weight: 80, category: 'politician' },
    { pattern: srbWord('gojković'), weight: 70, category: 'politician' },
    { pattern: /maja\s+gojković/i, weight: 75, category: 'politician' },
    { pattern: srbWord('gašić'), weight: 75, category: 'politician' },
    { pattern: /bratislav\s+gašić/i, weight: 80, category: 'politician' },
    { pattern: srbWord('vulin'), weight: 80, category: 'politician' },
    { pattern: /aleksandar\s+vulin/i, weight: 85, category: 'politician' },
    { pattern: srbWord('stefanović'), weight: 65, category: 'politician', context: ['ministar', 'nebojša', 'vlad'] },
    { pattern: /nebojša\s+stefanović/i, weight: 75, category: 'politician' },
    { pattern: srbWord('lončar'), weight: 60, category: 'politician', context: ['ministar', 'zlatibor', 'zdravlj'] },
    { pattern: srbWord('đorđević'), weight: 55, category: 'politician', context: ['ministar', 'zoran'] },
    { pattern: srbWord('mihajlović'), weight: 55, category: 'politician', context: ['zorana', 'ministar'] },
    // Minister of Mining and Energy
    { pattern: srbWord('đedović'), weight: 80, category: 'politician' },
    { pattern: /dubravka\s+đedović/i, weight: 85, category: 'politician' },
    { pattern: /đedović\s+handanović/i, weight: 85, category: 'politician' },
    { pattern: /ministar(?:ka)?\s+(?:rudarstva|energetike)/i, weight: 60, category: 'politician', context: ['đedović', 'izjav', 'rekla', 'najav'] },
    
    // Parliament and party officials
    { pattern: srbWord('orlić'), weight: 65, category: 'politician' },
    { pattern: /vladimir\s+orlić/i, weight: 70, category: 'politician' },
    { pattern: srbWord('martinović'), weight: 60, category: 'politician', context: ['aleksandar', 'sns', 'poslanik'] },
    
    // Close business/media allies
    { pattern: srbWord('mitrovića?'), weight: 50, category: 'oligarch', context: ['željko', 'pink', 'tv'] },
    { pattern: /željko\s+mitrović/i, weight: 70, category: 'oligarch' }
  ],

  // Tier 3: Pro-regime media ecosystem
  tier3: [
    // TV channels known for propaganda
    { pattern: /\bpink\b/i, weight: 45, category: 'media', context: ['tv', 'emisija', 'kanal', 'televizij'] },
    { pattern: /\bhappy\b/i, weight: 45, category: 'media', context: ['tv', 'kanal', 'televizij'] },
    { pattern: /\brts\b/i, weight: 35, category: 'media', context: ['dnevnik', 'emisij', 'izjav'] },
    { pattern: /\bb92\b/i, weight: 30, category: 'media', context: ['vest', 'izjav', 'prenosi'] },
    
    // Tabloids
    { pattern: /\binformer\b/i, weight: 70, category: 'tabloid' },
    { pattern: /\balo[\s!]?\b/i, weight: 55, category: 'tabloid', context: ['novine', 'portal', 'medij', 'piše', 'prenosi'] },
    { pattern: /\bsrpski\s+telegraf\b/i, weight: 65, category: 'tabloid' },
    { pattern: /\bkurir\b/i, weight: 50, category: 'tabloid' },
    { pattern: /\bobjektiv\b/i, weight: 55, category: 'tabloid' },
    { pattern: /\brepublika\b/i, weight: 40, category: 'tabloid', context: ['portal', 'medij', 'piše'] },
    { pattern: /\bespreso\b/i, weight: 40, category: 'tabloid' }
  ],

  // Tier 4: Propaganda phrases and patterns
  tier4: [
    // Cult of personality patterns
    { pattern: /predsednik\s+(poručio|najavio|obećao|rekao|izjavio)/i, weight: 60, category: 'propaganda' },
    { pattern: /vučić\s+(poručio|najavio|obećao|upozorio|otkrio)/i, weight: 70, category: 'propaganda' },
    { pattern: /lideri?\s+(sns|naprednjak)/i, weight: 55, category: 'propaganda' },
    
    // Demonization of critics/opposition
    { pattern: /stran[ie]\s+(služb|plaćeni|agent|mercenar)/i, weight: 60, category: 'demonization' },
    { pattern: /strani\s+faktor/i, weight: 65, category: 'demonization' },
    { pattern: srbWord('izdajni[ck]'), weight: 50, category: 'demonization' },
    { pattern: /rušen?je?\s+(srbi|držav|vlast)/i, weight: 55, category: 'demonization' },
    { pattern: /destabilizacij/i, weight: 50, category: 'demonization' },
    { pattern: srbWord('prevrat'), weight: 55, category: 'demonization' },
    { pattern: srbWord('puč'), weight: 55, category: 'demonization' },
    
    // Empty political theater
    { pattern: /sastanak\s+(sa|s)\s+/i, weight: 35, category: 'theater', context: ['vučić', 'predsednik', 'brnabić', 'vlad'] },
    { pattern: /poseta\s+/i, weight: 30, category: 'theater', context: ['vučić', 'predsednik', 'premijer'] },
    { pattern: /potpisivanje\s+(sporazum|ugovor)/i, weight: 35, category: 'theater', context: ['vučić', 'vlad', 'srbij'] },
    { pattern: /srbija\s+(napreduje|raste|jača)/i, weight: 50, category: 'propaganda' },
    
    // Fear/crisis mongering
    { pattern: srbWord('haos'), weight: 35, category: 'fear', context: ['opozicij', 'protest', 'napad'] },
    { pattern: /ugrožava/i, weight: 40, category: 'fear', context: ['srbij', 'držav', 'mir', 'stabilnost'] },
    { pattern: /napad\s+na\s+(vučić|predsedni|srbij)/i, weight: 55, category: 'fear' },
    
    // Glittering generalities
    { pattern: srbWord('stabilnost'), weight: 30, category: 'generality', context: ['vučić', 'vlad', 'sns', 'srbij'] },
    { pattern: srbWord('napredak'), weight: 30, category: 'generality', context: ['vučić', 'vlad', 'sns', 'srbij'] },
    { pattern: srbWord('budućnost'), weight: 25, category: 'generality', context: ['vučić', 'vlad', 'sns', 'obećao'] },
    
    // Controversial topics
    { pattern: /rio\s+tinto/i, weight: 50, category: 'controversy' },
    { pattern: /ekspo\s*20\d{2}/i, weight: 55, category: 'controversy' },
    { pattern: srbWord('litijum'), weight: 45, category: 'controversy' },
    { pattern: srbWord('jadar'), weight: 40, category: 'controversy', context: ['rudnik', 'projekat', 'rio'] },
    { pattern: /belgrade\s*waterfront/i, weight: 45, category: 'controversy' },
    { pattern: /beograd\s*na\s*vodi/i, weight: 45, category: 'controversy' },
    { pattern: srbWord('kosovo'), weight: 35, category: 'controversy', context: ['pregovor', 'dijalog', 'vučić', 'srbij'] }
  ],

  // Tier 5: Opposition/sensationalism (filter political theater about opposition too)
  tier5: [
    // Opposition figures (still political noise)
    { pattern: srbWord('đilas'), weight: 50, category: 'opposition' },
    { pattern: /dragan\s+đilas/i, weight: 55, category: 'opposition' },
    { pattern: srbWord('jeremić'), weight: 45, category: 'opposition' },
    { pattern: /vuk\s+jeremić/i, weight: 50, category: 'opposition' },
    { pattern: srbWord('obradović'), weight: 40, category: 'opposition', context: ['boško', 'dveri', 'opozicij'] },
    { pattern: /boško\s+obradović/i, weight: 50, category: 'opposition' },
    { pattern: srbWord('šapić'), weight: 40, category: 'opposition' },
    { pattern: /marinika\s+tepić/i, weight: 45, category: 'opposition' },
    { pattern: /radomir\s+lazović/i, weight: 40, category: 'opposition' },
    { pattern: /dobrica\s+veselinović/i, weight: 40, category: 'opposition' },
    
    // Coalition/party names
    { pattern: /srbija\s+protiv\s+nasilj/i, weight: 50, category: 'opposition' },
    { pattern: srbWord('proglasi'), weight: 35, category: 'opposition', context: ['opozicij', 'protest'] },
    
    // Sensationalist clickbait patterns
    { pattern: srbWord('šok(?:antno|irao|irala)?'), weight: 35, category: 'clickbait' },
    { pattern: srbWord('drama'), weight: 30, category: 'clickbait', context: ['politi', 'skupštin', 'vlad'] },
    { pattern: srbWord('skandal'), weight: 35, category: 'clickbait', context: ['politi', 'vlad', 'ministar'] },
    { pattern: srbWord('obračun'), weight: 35, category: 'clickbait', context: ['politi', 'stranke', 'lideri'] },
    { pattern: srbWord('žestok[oa]?'), weight: 30, category: 'clickbait', context: ['rasprav', 'sukob', 'reakcij'] }
  ],

  // Whitelist - ONLY truly important investigative journalism
  whitelist: [
    { pattern: /hapšenj[ea]/i, context: ['korupcij', 'kriminal', 'policij'] },
    { pattern: /presud[aeu]/i, context: ['sud', 'zatvor', 'osuđen'] },
    { pattern: /istraga/i, context: ['tužilaštv', 'policij', 'kriminal'] },
    { pattern: /žrtv[ae]/i, context: ['poginu', 'nasilje', 'nesreć'] },
    { pattern: /novi\s*sad/i, context: ['tragedij', 'žrtv', 'nadstrešnic'] },
    { pattern: /nadstrešnic/i, weight: -100, category: 'important' }
  ]
};

// ========== CONTENT HASHING ==========
// SHA-256 for content-based cache keys
async function sha256(text) {
  const encoder = new TextEncoder();
  const data = encoder.encode(text);
  const hashBuffer = await crypto.subtle.digest('SHA-256', data);
  const hashArray = Array.from(new Uint8Array(hashBuffer));
  return hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
}

function normalizeText(text) {
  return text.toLowerCase().trim().replace(/\s+/g, ' ');
}

async function generateContentHash(content) {
  const normalized = normalizeText(
    `${content.title}\n${content.description || ''}`
  );
  return sha256(normalized);
}

// ========== SEMANTIC CLASSIFIER ==========
// Calls the backend API for embedding-based classification
class SemanticClassifier {
  constructor(config = {}) {
    this.apiUrl = config.apiUrl || API_URL;
    this.batchSize = config.batchSize || 50;
    this.enabled = config.enabled !== false;
    this.localCache = new Map();
    this.pendingBatch = [];
    this.batchTimeout = null;
    // Stats tracking
    this.stats = {
      totalProcessed: 0,
      blocked: 0,
      allowed: 0,
      cached: 0,
      errors: 0
    };
  }

  getStats() {
    return { ...this.stats };
  }

  async classifyBatch(articles, adapter) {
    if (!this.enabled || articles.length === 0) return;

    console.log(`[Šumski] Semantic classification for ${articles.length} articles`);

    // Prepare batch with content hashes
    const batch = [];
    for (const { element, content, score } of articles) {
      const hash = await generateContentHash(content);

      // Check local session cache
      if (this.localCache.has(hash)) {
        const cached = this.localCache.get(hash);
        this.applyResult(element, cached, adapter);
        this.stats.cached++;
        continue;
      }

      batch.push({
        element,
        hash,
        contentHash: hash,
        url: window.location.href,
        title: content.title,
        description: content.description || '',
        keywordScore: score
      });

      element.dataset.sumskiHash = hash;
      element.dataset.sumskiStatus = 'pending';
    }

    if (batch.length === 0) return;

    // Split into chunks of batchSize (default 50)
    const chunks = [];
    for (let i = 0; i < batch.length; i += this.batchSize) {
      chunks.push(batch.slice(i, i + this.batchSize));
    }

    console.log(`[Šumski] Processing ${batch.length} articles in ${chunks.length} batch(es)`);

    // Process each chunk
    for (const chunk of chunks) {
      await this.processChunk(chunk, adapter);
    }

    // Log final stats
    this.logStats();
  }

  async processChunk(batch, adapter) {
    // Send batch request
    try {
      const response = await fetch(`${this.apiUrl}/api/classify/batch`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          items: batch.map(({ contentHash, url, title, description }) => ({
            contentHash, url, title, description
          }))
        })
      });

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        console.error('[Šumski] API error response:', JSON.stringify(errorData));
        throw new Error(`HTTP ${response.status}: ${errorData.error || 'Unknown error'}`);
      }

      const data = await response.json();
      
      // Track stats from API response
      this.stats.cached += data.cached || 0;
      this.stats.errors += data.errors || 0;

      // Apply results and count classifications
      for (const item of batch) {
        const result = data.results[item.hash];
        if (result) {
          this.stats.totalProcessed++;
          this.stats[result.classification]++;
          this.applyResult(item.element, result, adapter);
          this.localCache.set(item.hash, result);
        } else {
          // No result from API - default to allowed (keep visible)
          item.element.dataset.sumskiStatus = 'complete';
          item.element.dataset.sumskiClassification = 'allowed';
          this.stats.allowed++;
        }
      }

    } catch (error) {
      console.error('[Šumski] Semantic classification failed:', error);
      this.stats.errors += batch.length;
      // On error, mark articles as error state (keep visible)
      for (const { element } of batch) {
        element.dataset.sumskiStatus = 'error';
      }
    }
  }

  logStats() {
    const s = this.stats;
    console.log(
      `%c[Šumski AI] 🤖 Semantic Analysis Complete%c\n` +
      `  ✅ Allowed: ${s.allowed}  |  � Blocked: ${s.blocked}\n` +
      `  📊 Total: ${s.totalProcessed}  |  💾 Cached: ${s.cached}  |  ⚠️ Errors: ${s.errors}`,
      'color: #4CAF50; font-weight: bold; font-size: 12px;',
      'color: #666; font-size: 11px;'
    );
  }

  applyResult(element, result, adapter) {
    element.dataset.sumskiStatus = 'complete';
    element.dataset.sumskiClassification = result.classification;

    if (result.classification === 'blocked') {
      const imageUrl = NATURE_IMAGES[Math.floor(Math.random() * NATURE_IMAGES.length)];
      adapter.replaceArticle(element, {
        imageUrl,
        showLabel: true
      });
      console.log(`[Šumski] Semantic blocked: "${element.dataset.sumskiHash?.slice(0, 8)}..." (score: ${result.score})`);
    }
    // 'allowed' and 'uncertain' articles remain visible
  }
}

// ========== TOXICITY SCORER ==========
class ToxicityScorer {
  constructor(config = {}) {
    this.config = {
      threshold: config.threshold || 70,
      enableControversialTopics: config.enableControversialTopics !== false,
      enableOppositionFilter: config.enableOppositionFilter !== false,
      ...config
    };
  }

  analyzeArticle(content) {
    const { title = '', description = '', body = '' } = content;
    const fullText = `${title} ${description} ${body}`;

    let score = 0;
    const matches = [];

    // Check whitelist first - important news should not be blocked
    if (this.isWhitelisted(fullText)) {
      return { score: 0, matches: [], shouldBlock: false, reason: 'whitelisted', confidence: 1.0 };
    }

    // Tier 1: Core regime - immediate block on title/description match
    const tier1Result = this.checkTier(KEYWORDS_DB.tier1, title, description);
    if (tier1Result.matches.length > 0) {
      return { score: 100, matches: tier1Result.matches, shouldBlock: true, reason: 'tier1_match', confidence: 0.95 };
    }

    // Tier 2: Extended regime network
    const tier2Result = this.checkTier(KEYWORDS_DB.tier2, title, description, body);
    score += tier2Result.score;
    matches.push(...tier2Result.matches);

    // Tier 3: Pro-regime media
    const tier3Result = this.checkTier(KEYWORDS_DB.tier3, title, description, body);
    score += tier3Result.score;
    matches.push(...tier3Result.matches);

    // Tier 4: Propaganda phrases & controversial topics
    if (this.config.enableControversialTopics) {
      const tier4Result = this.checkTier(KEYWORDS_DB.tier4, title, description, body);
      score += tier4Result.score;
      matches.push(...tier4Result.matches);
    }

    // Tier 5: Opposition & sensationalism filter
    if (this.config.enableOppositionFilter) {
      const tier5Result = this.checkTier(KEYWORDS_DB.tier5, title, description, body);
      score += tier5Result.score;
      matches.push(...tier5Result.matches);
    }

    // Apply boosters
    score = this.applyContextBoost(score, matches);
    score = this.applyTitleBoost(score, matches);
    
    score = Math.min(score, 100);
    const shouldBlock = score >= this.config.threshold;

    return { 
      score: Math.round(score), 
      matches, 
      shouldBlock, 
      reason: shouldBlock ? 'score_threshold' : 'below_threshold',
      confidence: this.calculateConfidence(score, matches)
    };
  }

  // Boost score if multiple propaganda categories detected
  applyContextBoost(score, matches) {
    if (matches.length === 0) return score;
    const categories = new Set(matches.map(m => m.category));
    // Multiple categories = likely coordinated content
    if (categories.size >= 3) score *= 1.25;
    else if (categories.size >= 2) score *= 1.1;
    return score;
  }

  // Title matches are more significant
  applyTitleBoost(score, matches) {
    const titleMatches = matches.filter(m => m.location === 'title');
    if (titleMatches.length > 0) score *= 1.15;
    return score;
  }

  calculateConfidence(score, matches) {
    if (score >= 90) return 0.95;
    if (score >= 70) return 0.85;
    if (score >= 50) return 0.70;
    return 0.55;
  }

  checkTier(tier, ...texts) {
    let tierScore = 0;
    const matches = [];

    tier.forEach(keyword => {
      texts.forEach((text, index) => {
        if (!text) return;
        if (keyword.pattern.test(text)) {
          if (keyword.context) {
            const hasContext = keyword.context.some(ctx => new RegExp(ctx, 'i').test(text));
            if (!hasContext) return;
          }
          tierScore += keyword.weight;
          matches.push({
            keyword: keyword.pattern.source,
            weight: keyword.weight,
            category: keyword.category,
            location: ['title', 'description', 'body'][index] || 'unknown'
          });
        }
      });
    });

    return { score: tierScore, matches };
  }

  isWhitelisted(fullText) {
    return KEYWORDS_DB.whitelist.some(item => {
      if (!item.pattern.test(fullText)) return false;
      if (item.context) {
        return item.context.some(ctx => new RegExp(ctx, 'i').test(fullText));
      }
      return true;
    });
  }
}

// ========== TOXIC DETECTOR ==========
class ToxicDetector {
  constructor(config = {}) {
    this.config = { threshold: 70, ...config };
    this.scorer = new ToxicityScorer(this.config);
    this.stats = { totalScanned: 0, totalBlocked: 0, byCategory: {} };
  }

  detect(content) {
    this.stats.totalScanned++;
    const result = this.scorer.analyzeArticle(content);
    if (result.shouldBlock) {
      this.stats.totalBlocked++;
      result.matches.forEach(match => {
        this.stats.byCategory[match.category] = (this.stats.byCategory[match.category] || 0) + 1;
      });
      // Update badge count
      this.updateBadge();
    }
    return result;
  }

  updateBadge() {
    try {
      chrome.runtime.sendMessage({ 
        action: 'updateBadge', 
        count: this.stats.totalBlocked 
      });
    } catch (e) {
      // Ignore errors if background script not ready
    }
  }

  getStats() {
    return {
      ...this.stats,
      blockRate: this.stats.totalScanned > 0
        ? Math.round((this.stats.totalBlocked / this.stats.totalScanned) * 100)
        : 0
    };
  }

  updateConfig(newConfig) {
    this.config = { ...this.config, ...newConfig };
    this.scorer = new ToxicityScorer(this.config);
  }
}

// ========== SITE ADAPTER ==========
class N1InfoAdapter {
  constructor() {
    this.id = 'n1info';
    this.name = 'N1 Info';
    this.processed = new Set();
    this.replacements = new Map(); // Store replacement -> original mapping
  }

  findArticles() {
    // N1 uses various article containers
    const selectors = 'article, [class*="article"], [class*="post-item"], a[href*="/vesti/"]';
    const elements = document.querySelectorAll(selectors);

    return Array.from(elements).filter(el => {
      // Skip if manually unhidden by user
      if (el.getAttribute('data-sumski-filter-unhidden') === 'true') {
        return false;
      }
      // Check if article ID is in unhidden set
      if (typeof unhiddenArticles !== 'undefined') {
        const articleId = this.getArticleId(el);
        if (unhiddenArticles.has(articleId)) {
          return false;
        }
      }
      // Must have some text content
      const text = el.textContent.trim();
      return text.length > 20 && !el.closest('[class*="ad"]') && !this.isProcessed(el);
    });
  }

  extractContent(element) {
    const getText = (selectors) => {
      for (const sel of selectors) {
        const el = element.querySelector(sel);
        if (el) return el.textContent.trim();
      }
      return element.textContent.trim().substring(0, 500);
    };

    return {
      title: getText(['h1', 'h2', 'h3', '[class*="title"]', '[class*="headline"]']),
      description: getText(['.excerpt', '.summary', 'p']),
      body: element.textContent.substring(0, 1000)
    };
  }

  createReplacement(element, data) {
    const rect = element.getBoundingClientRect();
    const height = Math.max(Math.min(rect.height, 600), 200);

    const div = document.createElement('div');
    div.className = 'sumski-filter-replacement';
    div.style.cssText = `
      width: 100%;
      height: ${height}px;
      display: flex;
      align-items: center;
      justify-content: center;
      position: relative;
      background: #f7f5f0;
      border-radius: 8px;
      overflow: hidden;
      box-shadow: 0 2px 8px rgba(45,59,49,0.1);
    `;

    const img = document.createElement('img');
    img.src = data.imageUrl;
    img.style.cssText = 'width: 100%; height: 100%; object-fit: cover; position: absolute; top: 0; left: 0;';
    div.appendChild(img);

    if (data.showLabel) {
      const label = document.createElement('div');
      label.style.cssText = `
        position: absolute;
        bottom: 0;
        left: 0;
        right: 0;
        padding: 12px;
        background: linear-gradient(to top, rgba(45,59,49,0.85), transparent);
        color: #f7f5f0;
        text-align: center;
        font-size: 14px;
        font-weight: 500;
        font-family: 'Inter', system-ui, sans-serif;
        z-index: 1;
      `;
      // Forest-themed positive messages
      const messages = [
        '🌲 Preskočili smo ovu vest za vas',
        '🌿 Fokusirajte se na ono što možete da promenite',
        '🍃 Vaša energija je sačuvana',
        '🌳 Uživajte u tišini šume',
        '🌲 Manje buke, više mira',
        '🍂 Ova vest nije vredna vaše pažnje',
        '🌿 Idemo dalje, ka lepšim stvarima',
        '🌲 Šuma filtrira buku'
      ];
      label.textContent = messages[Math.floor(Math.random() * messages.length)];
      div.appendChild(label);
    }

    return div;
  }

  replaceArticle(element, data) {
    const replacement = this.createReplacement(element, data);

    // Clone the original element before replacing
    const originalClone = element.cloneNode(true);
    this.replacements.set(replacement, originalClone);

    element.style.transition = 'opacity 300ms';
    element.style.opacity = '0';

    setTimeout(() => {
      element.replaceWith(replacement);
      replacement.style.opacity = '0';
      replacement.style.transition = 'opacity 300ms';
      setTimeout(() => replacement.style.opacity = '1', 10);
    }, 300);

    this.markAsProcessed(element);
  }

  unhideAll() {
    console.log(`[Šumski Filter] Unhiding ${this.replacements.size} articles...`);

    const replacements = Array.from(document.querySelectorAll('.sumski-filter-replacement'));
    let unhidden = 0;

    replacements.forEach(replacement => {
      const original = this.replacements.get(replacement);
      if (original) {
        // Mark the original element so it won't be re-processed
        original.setAttribute('data-sumski-filter-unhidden', 'true');

        // Store unhidden article ID for persistent tracking
        const articleId = this.getArticleId(original);
        if (typeof unhiddenArticles !== 'undefined') {
          unhiddenArticles.add(articleId);
        }

        replacement.style.transition = 'opacity 400ms';
        replacement.style.opacity = '0';

        setTimeout(() => {
          replacement.replaceWith(original);
          original.style.opacity = '0';
          original.style.transition = 'opacity 400ms';
          setTimeout(() => original.style.opacity = '1', 50);
        }, 400);

        this.replacements.delete(replacement);
        unhidden++;
      }
    });

    console.log(`[Šumski Filter] Unhidden ${unhidden} articles - they will stay visible`);
    return unhidden;
  }

  isProcessed(element) {
    return this.processed.has(this.getArticleId(element));
  }

  markAsProcessed(element) {
    this.processed.add(this.getArticleId(element));
  }

  getArticleId(element) {
    if (element.id) return element.id;
    const text = element.textContent.substring(0, 100);
    let hash = 0;
    for (let i = 0; i < text.length; i++) {
      hash = ((hash << 5) - hash) + text.charCodeAt(i);
      hash = hash & hash;
    }
    return hash.toString(36);
  }
}

// ========== MAIN ==========
let detector = null;
let adapter = null;
let semanticClassifier = null;
let settings = {};
let unhiddenArticles = new Set(); // Track manually unhidden articles

function init() {
  console.log('[Šumski Filter] Initializing...');

  adapter = new N1InfoAdapter();
  console.log(`[Šumski Filter] Using adapter: ${adapter.name}`);

  // Initialize semantic classifier
  if (SEMANTIC_ENABLED) {
    semanticClassifier = new SemanticClassifier({
      apiUrl: API_URL,
      enabled: true
    });
    console.log('[Šumski Filter] Semantic classification enabled');
  }

  chrome.storage.sync.get(['settings'], (data) => {
    settings = data.settings || {
      enabled: true,
      detectionThreshold: 70,
      enableControversialTopics: true,
      showLabel: true,
      semanticEnabled: SEMANTIC_ENABLED
    };

    detector = new ToxicDetector({
      threshold: settings.detectionThreshold,
      enableControversialTopics: settings.enableControversialTopics
    });

    if (settings.enabled) {
      scanPage();
      setupObserver();
    }
  });

  chrome.storage.onChanged.addListener((changes) => {
    if (changes.settings) {
      settings = changes.settings.newValue;
      if (detector) {
        detector.updateConfig({
          threshold: settings.detectionThreshold,
          enableControversialTopics: settings.enableControversialTopics
        });
      }
      if (semanticClassifier) {
        semanticClassifier.enabled = settings.semanticEnabled !== false;
      }
    }
  });
}

async function scanPage() {
  if (!adapter || !detector) return;

  const articles = adapter.findArticles();
  console.log(`[Šumski Filter] Found ${articles.length} articles to process`);

  const semanticCandidates = [];

  for (const article of articles) {
    const content = adapter.extractContent(article);
    if (!content.title) continue;

    const result = detector.detect(content);

    // Log every article's score for debugging
    console.log(`[Šumski] Score ${result.score}: "${content.title.substring(0, 40)}..."`);

    if (result.shouldBlock) {
      // Immediate block based on keywords (high confidence)
      const imageUrl = NATURE_IMAGES[Math.floor(Math.random() * NATURE_IMAGES.length)];
      adapter.replaceArticle(article, {
        imageUrl,
        showLabel: settings.showLabel
      });
      console.log(`[Šumski Filter] Blocked: "${content.title.substring(0, 50)}..." (score: ${result.score})`);
    } else if (result.score < KEYWORD_BLOCK_THRESHOLD) {
      // Not blocked by keywords - send to semantic classification
      // This catches articles with score 0 (no keyword match) that might still be propaganda
      semanticCandidates.push({
        element: article,
        content,
        score: result.score
      });
      article.dataset.sumskiScore = result.score;
      article.dataset.sumskiStatus = 'pending-semantic';
      console.log(`[Šumski] Queuing for semantic: "${content.title.substring(0, 40)}..." (keyword score: ${result.score})`);
    }
  }

  const stats = detector.getStats();
  console.log(`[Šumski Filter] Stats: ${stats.totalBlocked}/${stats.totalScanned} blocked (${stats.blockRate}%)`);

  // Process uncertain articles with semantic classifier
  if (semanticClassifier && semanticCandidates.length > 0) {
    console.log(`[Šumski Filter] Sending ${semanticCandidates.length} articles for semantic classification`);
    // Run async - don't block the page
    semanticClassifier.classifyBatch(semanticCandidates, adapter).catch(err => {
      console.error('[Šumski Filter] Semantic classification error:', err);
    });
  }
}

function setupObserver() {
  const observer = new MutationObserver(() => {
    setTimeout(scanPage, 500);
  });

  observer.observe(document.body, {
    childList: true,
    subtree: true
  });
}

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'getStats' && detector) {
    const stats = detector.getStats();
    // Add semantic/AI stats
    if (semanticClassifier) {
      stats.semanticEnabled = semanticClassifier.enabled;
      stats.semanticStats = semanticClassifier.getStats();
    } else {
      stats.semanticEnabled = false;
      stats.semanticStats = { totalProcessed: 0, blocked: 0, allowed: 0, cached: 0, errors: 0 };
    }
    sendResponse({ stats });
  } else if (request.action === 'unhideAll' && adapter) {
    const count = adapter.unhideAll();
    sendResponse({ success: true, count: count });
  } else if (request.action === 'clearUnhidden') {
    // Clear the unhidden tracking set
    unhiddenArticles.clear();
    // Remove attributes from all unhidden articles
    document.querySelectorAll('[data-sumski-filter-unhidden="true"]').forEach(el => {
      el.removeAttribute('data-sumski-filter-unhidden');
    });
    console.log('[Šumski Filter] Cleared unhidden list - blocking re-enabled');
    sendResponse({ success: true });
  } else if (request.action === 'toggleSemantic') {
    if (semanticClassifier) {
      semanticClassifier.enabled = request.enabled;
      console.log(`[Šumski Filter] Semantic classification ${request.enabled ? 'enabled' : 'disabled'}`);
    }
    sendResponse({ success: true });
  } else if (request.action === 'getArticleForLabeling') {
    // Get article data for labeling from context menu
    const article = getArticleFromContext(request.linkUrl, request.selectionText, request.pageUrl);
    sendResponse({ article });
  } else if (request.action === 'showNotification') {
    // Show notification toast
    showLabelNotification(request.message);
    sendResponse({ success: true });
  }
  return true;
});

// Get article data from the current page for labeling
function getArticleFromContext(linkUrl, selectionText, pageUrl) {
  // If we have a link URL, try to find that article
  if (linkUrl) {
    const link = document.querySelector(`a[href="${linkUrl}"], a[href="${new URL(linkUrl).pathname}"]`);
    if (link) {
      const article = link.closest('article, .article, [class*="article"], [class*="story"], [class*="card"]');
      if (article) {
        const title = article.querySelector('h1, h2, h3, h4, .title, [class*="title"]')?.textContent?.trim();
        const desc = article.querySelector('p, .description, .excerpt, [class*="desc"]')?.textContent?.trim();
        if (title) {
          return { title, description: desc || null, url: linkUrl };
        }
      }
    }
  }

  // If we have selection text, use that as the title
  if (selectionText && selectionText.length > 10) {
    return {
      title: selectionText.substring(0, 500),
      description: null,
      url: pageUrl || window.location.href
    };
  }

  // Fall back to page title
  const pageTitle = document.querySelector('h1.title, h1.article-title, article h1, .article-title, h1')?.textContent?.trim()
    || document.title;

  const pageDesc = document.querySelector('meta[name="description"]')?.getAttribute('content')
    || document.querySelector('article p, .article-body p')?.textContent?.trim();

  return {
    title: pageTitle,
    description: pageDesc || null,
    url: pageUrl || window.location.href
  };
}

// Show notification toast for labeling feedback
function showLabelNotification(message) {
  // Remove any existing notification
  const existing = document.getElementById('sumski-filter-notification');
  if (existing) existing.remove();

  const notification = document.createElement('div');
  notification.id = 'sumski-filter-notification';
  notification.textContent = message;
  notification.style.cssText = `
    position: fixed;
    bottom: 20px;
    right: 20px;
    background: #3d4f3f;
    color: #f5f2eb;
    padding: 12px 20px;
    border-radius: 8px;
    font-size: 14px;
    font-family: -apple-system, BlinkMacSystemFont, sans-serif;
    box-shadow: 0 4px 12px rgba(0,0,0,0.3);
    z-index: 999999;
    animation: sumskiFadeIn 0.3s ease;
  `;

  // Add animation styles if not already added
  if (!document.getElementById('sumski-filter-styles')) {
    const style = document.createElement('style');
    style.id = 'sumski-filter-styles';
    style.textContent = `
      @keyframes sumskiFadeIn {
        from { opacity: 0; transform: translateY(10px); }
        to { opacity: 1; transform: translateY(0); }
      }
      @keyframes sumskiFadeOut {
        from { opacity: 1; transform: translateY(0); }
        to { opacity: 0; transform: translateY(10px); }
      }
    `;
    document.head.appendChild(style);
  }

  document.body.appendChild(notification);

  // Auto-remove after 3 seconds
  setTimeout(() => {
    notification.style.animation = 'sumskiFadeOut 0.3s ease';
    setTimeout(() => notification.remove(), 300);
  }, 3000);
}

if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', init);
} else {
  init();
}
