/**
 * Popup UI Logic - Simplified Version
 * 
 * Removed user-facing controls for threshold and controversial topics.
 * Keywords run locally as first pass, semantic AI handles the rest.
 */

// Load settings and stats
function loadSettings() {
  chrome.storage.sync.get(['settings'], (data) => {
    const settings = data.settings || {
      enabled: true,
      // Keep these defaults but don't expose to UI
      detectionThreshold: 70,
      enableControversialTopics: true,
      showLabel: true
    };

    document.getElementById('enabled').checked = settings.enabled;
    updateStatusIndicator(settings.enabled);
  });

  loadStats();
}

// Load stats from content script
function loadStats() {
  chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
    if (tabs[0]) {
      chrome.tabs.sendMessage(tabs[0].id, { action: 'getStats' }, (response) => {
        if (chrome.runtime.lastError) {
          updateStats({ totalBlocked: 0, totalScanned: 0 });
          return;
        }
        if (response && response.stats) {
          updateStats(response.stats);
        }
      });
    }
  });
}

// Update stats display
function updateStats(stats) {
  // Total blocked = keywords + AI
  const keywordBlocked = stats.totalBlocked || 0;
  const aiBlocked = stats.semanticStats?.blocked || 0;
  const totalBlocked = keywordBlocked + aiBlocked;
  
  document.getElementById('blocked').textContent = totalBlocked;
  document.getElementById('scanned').textContent = stats.totalScanned || 0;
  
  // Update badge
  const badge = document.getElementById('logoBadge');
  badge.textContent = totalBlocked;
  badge.style.display = totalBlocked > 0 ? 'flex' : 'none';

  // AI status
  const aiStatusEl = document.getElementById('aiStatus');
  const aiDot = document.getElementById('aiDot');
  const aiText = document.getElementById('aiText');
  const semantic = stats.semanticStats;
  
  if (stats.semanticEnabled && semantic && semantic.totalProcessed > 0) {
    aiStatusEl.classList.remove('hidden');
    
    if (semantic.errors > 0) {
      aiDot.classList.add('error');
      aiText.textContent = `⚠️ AI: ${semantic.errors} greška`;
    } else {
      aiDot.classList.remove('error');
      const cached = semantic.cached || 0;
      if (cached > 0) {
        aiText.textContent = `🤖 AI analizirao ${semantic.totalProcessed} članaka (${cached} iz keša)`;
      } else {
        aiText.textContent = `🤖 AI analizirao ${semantic.totalProcessed} članaka`;
      }
    }
  } else {
    aiStatusEl.classList.add('hidden');
  }
}

// Update status indicator
function updateStatusIndicator(enabled) {
  const indicator = document.getElementById('statusIndicator');
  if (enabled) {
    indicator.classList.remove('inactive');
  } else {
    indicator.classList.add('inactive');
  }
}

// Save settings
function saveSettings() {
  // Get current settings first to preserve hidden values
  chrome.storage.sync.get(['settings'], (data) => {
    const current = data.settings || {};
    const settings = {
      enabled: document.getElementById('enabled').checked,
      // Preserve these values (not exposed in UI)
      detectionThreshold: current.detectionThreshold || 70,
      enableControversialTopics: current.enableControversialTopics !== false,
      showLabel: current.showLabel !== false
    };

    chrome.storage.sync.set({ settings }, () => {
      console.log('[Šumski Filter] Settings saved');
      updateStatusIndicator(settings.enabled);
    });
  });
}

// Unhide all blocked articles
function unhideArticles() {
  const btn = document.getElementById('unhideBtn');
  const originalHTML = btn.innerHTML;

  chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
    if (tabs[0]) {
      chrome.tabs.sendMessage(tabs[0].id, { action: 'unhideAll' }, (response) => {
        if (chrome.runtime.lastError) {
          console.error('[Šumski Filter] Could not unhide:', chrome.runtime.lastError);
          return;
        }
        if (response && response.success) {
          btn.classList.add('btn-success');
          btn.innerHTML = `✓ Prikazano ${response.count || 0} vesti`;

          setTimeout(() => {
            btn.classList.remove('btn-success');
            btn.innerHTML = originalHTML;
            loadStats();
          }, 5000);
        }
      });
    }
  });
}

// Refresh current tab
function refreshPage() {
  chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
    if (tabs[0]) {
      chrome.tabs.reload(tabs[0].id);
      window.close();
    }
  });
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
  const enabledEl = document.getElementById('enabled');
  const unhideBtn = document.getElementById('unhideBtn');
  const refreshBtn = document.getElementById('refreshBtn');

  if (enabledEl) enabledEl.addEventListener('change', saveSettings);
  if (unhideBtn) unhideBtn.addEventListener('click', unhideArticles);
  if (refreshBtn) refreshBtn.addEventListener('click', refreshPage);

  loadSettings();
  setInterval(loadStats, 2000);
});
